/* geoclue-stumbler-application.c
 *
 * Copyright 2024 Christopher Talbot
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

#include "config.h"
#include "version.h"

#include <glib/gi18n.h>

#include "geoclue-stumbler-application.h"
#include "geoclue-stumbler-debug-info.h"
#include "geoclue-stumbler-preferences-dialog.h"
#include "geoclue-stumbler-settings.h"
#include "geoclue-stumbler-window.h"

struct _GeoclueStumblerApplication
{
	AdwApplication parent_instance;
    gboolean app_debug;

    GeoclueStumblerSettings *settings;
};

G_DEFINE_FINAL_TYPE (GeoclueStumblerApplication, geoclue_stumbler_application, ADW_TYPE_APPLICATION)

GeoclueStumblerApplication *
geoclue_stumbler_application_new (const char        *application_id,
                                  GApplicationFlags  flags)
{
	g_return_val_if_fail (application_id != NULL, NULL);

	return g_object_new (GEOCLUE_STUMBLER_TYPE_APPLICATION,
	                     "application-id", application_id,
	                     "flags", flags,
	                     NULL);
}

static int
geoclue_stumbler_application_handle_local_options (GApplication *application,
                                                   GVariantDict *options)
{
  GeoclueStumblerApplication *self = (GeoclueStumblerApplication *)application;
  if (g_variant_dict_contains (options, "version"))
    {
      g_print ("%s, git version: %s\n", PACKAGE_VERSION, PACKAGE_VCS_VERSION);
      return 0;
    }

  if (g_variant_dict_contains (options, "debug"))
    {
      g_setenv ("G_MESSAGES_DEBUG", "all", TRUE);
      g_log_set_debug_enabled (TRUE);
      self->app_debug = TRUE;
      g_debug ("Debug on");
    }

  return -1;
}

static void
geoclue_stumbler_application_activate (GApplication *app)
{
	GtkWindow *window;

	g_assert (GEOCLUE_STUMBLER_IS_APPLICATION (app));

	window = gtk_application_get_active_window (GTK_APPLICATION (app));

	if (window == NULL)
		window = g_object_new (GEOCLUE_STUMBLER_TYPE_WINDOW,
		                       "application", app,
		                       NULL);

	gtk_window_present (window);
}

static void
geoclue_stumbler_application_class_init (GeoclueStumblerApplicationClass *klass)
{
	GApplicationClass *app_class = G_APPLICATION_CLASS (klass);

	app_class->activate = geoclue_stumbler_application_activate;
    app_class->handle_local_options = geoclue_stumbler_application_handle_local_options;
}

static void
geoclue_stumbler_application_preferences_action (GSimpleAction *action,
                                                 GVariant      *parameter,
                                                 gpointer       user_data)
{
	GeoclueStumblerApplication *self = user_data;
    GeoclueStumblerPreferencesDialog *dialog = NULL;
	GtkWindow *window = NULL;

	g_assert (GEOCLUE_STUMBLER_IS_APPLICATION (self));

	window = gtk_application_get_active_window (GTK_APPLICATION (self));

    dialog = geoclue_stumbler_preferences_new ();

    adw_dialog_present (ADW_DIALOG(dialog), GTK_WIDGET (window));
}

static void
geoclue_stumbler_application_about_action (GSimpleAction *action,
                                           GVariant      *parameter,
                                           gpointer       user_data)
{
	static const char *developers[] = {"Christopher Talbot", NULL};
    static const char *artists[] = {"Yann Couturier", "Ethan Grobin", NULL};
	GeoclueStumblerApplication *self = user_data;
    g_autofree char *debug_info = NULL;
	GtkWindow *window = NULL;

	g_assert (GEOCLUE_STUMBLER_IS_APPLICATION (self));

	window = gtk_application_get_active_window (GTK_APPLICATION (self));

    debug_info = geoclue_stumbler_generate_debug_info ();

	adw_show_about_dialog (GTK_WIDGET (window),
	                       "application-name", GETTEXT_PACKAGE,
	                       "application-icon", PACKAGE_ID,
	                       "developer-name", "Christopher Talbot",
	                       "translator-credits", _("translator-credits"),
	                       "version", PACKAGE_VCS_VERSION,
	                       "developers", developers,
                           "artists", artists,
	                       "copyright", "© 2024 Christopher Talbot",
                           "website", "https://gitlab.com/kop316/geoclue-stumbler",
                           "issue-url", "https://gitlab.com/kop316/geoclue-stumbler/-/issues",
                           "support-url", "https://gitlab.com/kop316/geoclue-stumbler/-/wikis/home",
                           "debug-info", debug_info,
                           "license-type", GTK_LICENSE_GPL_3_0,
	                       NULL);
}

static void
geoclue_stumbler_application_quit_action (GSimpleAction *action,
                                          GVariant      *parameter,
                                          gpointer       user_data)
{
	GeoclueStumblerApplication *self = user_data;

	g_assert (GEOCLUE_STUMBLER_IS_APPLICATION (self));

	g_application_quit (G_APPLICATION (self));
}

static const GActionEntry app_actions[] = {
	{ "quit", geoclue_stumbler_application_quit_action },
	{ "about", geoclue_stumbler_application_about_action },
	{ "preferences", geoclue_stumbler_application_preferences_action },
};

static GOptionEntry cmd_options[] = {
  { "debug", 'd', 0, G_OPTION_ARG_NONE, G_OPTION_ARG_NONE,
    "Enable debugging output"},
  {
    "version", 'v', G_OPTION_FLAG_NONE, G_OPTION_ARG_NONE, NULL,
    ("Show release version"), NULL
  },
  { NULL }
};

static void
geoclue_stumbler_application_init (GeoclueStumblerApplication *self)
{
	g_action_map_add_action_entries (G_ACTION_MAP (self),
	                                 app_actions,
	                                 G_N_ELEMENTS (app_actions),
	                                 self);
	gtk_application_set_accels_for_action (GTK_APPLICATION (self),
	                                       "app.quit",
	                                       (const char *[]) { "<primary>q", NULL });

    g_application_add_main_option_entries (G_APPLICATION (self), cmd_options);

    self->app_debug = FALSE;

    self->settings = geoclue_stumbler_settings_get_default ();
}
