/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.storage;

import org.jboss.logging.Logger;
import org.keycloak.cluster.ClusterProvider;
import org.keycloak.cluster.ExecutionResult;
import org.keycloak.common.util.Time;
import org.keycloak.common.util.TriFunction;
import org.keycloak.component.ComponentModel;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.KeycloakSessionFactory;
import org.keycloak.models.ModelIllegalStateException;
import org.keycloak.models.RealmModel;
import org.keycloak.storage.UserStorageProvider;
import org.keycloak.storage.UserStorageProviderFactory;
import org.keycloak.storage.UserStorageProviderModel;
import org.keycloak.storage.user.ImportSynchronization;
import org.keycloak.storage.user.SynchronizationResult;
import org.keycloak.timer.ScheduledTask;
import org.keycloak.timer.TimerProvider;

final class UserStorageSyncTask
implements ScheduledTask {
    private static final Logger logger = Logger.getLogger(UserStorageSyncTask.class);
    private static final int TASK_EXECUTION_TIMEOUT = 30;
    private final String providerId;
    private final String realmId;
    private final UserStorageProviderModel.SyncMode syncMode;
    private final int period;

    UserStorageSyncTask(UserStorageProviderModel provider, UserStorageProviderModel.SyncMode syncMode) {
        this.providerId = provider.getId();
        this.realmId = provider.getParentId();
        this.syncMode = syncMode;
        this.period = UserStorageProviderModel.SyncMode.FULL.equals((Object)syncMode) ? provider.getFullSyncPeriod() : provider.getChangedSyncPeriod();
    }

    public void run(KeycloakSession session) {
        RealmModel realm = session.realms().getRealm(this.realmId);
        session.getContext().setRealm(realm);
        UserStorageProviderModel provider = this.getStorageModel(session);
        if (this.isSyncPeriod(provider)) {
            this.runWithResult(session);
            return;
        }
        logger.debugf("Ignored LDAP %s users-sync with storage provider %s due small time since last sync in realm %s", (Object)this.syncMode, (Object)provider.getName(), (Object)this.realmId);
    }

    public String getTaskName() {
        return UserStorageSyncTask.class.getSimpleName() + "-" + this.providerId + "-" + String.valueOf(this.syncMode);
    }

    SynchronizationResult runWithResult(KeycloakSession session) {
        try {
            return switch (this.syncMode) {
                default -> throw new IncompatibleClassChangeError();
                case UserStorageProviderModel.SyncMode.FULL -> this.runFullSync(session);
                case UserStorageProviderModel.SyncMode.CHANGED -> this.runIncrementalSync(session);
            };
        }
        catch (Throwable t) {
            logger.errorf(t, "Error occurred during %s users-sync in realm %s and user provider %s", (Object)this.syncMode, (Object)this.realmId, (Object)this.providerId);
            return SynchronizationResult.empty();
        }
    }

    boolean schedule(KeycloakSession session) {
        UserStorageProviderModel provider = this.getStorageModel(session);
        if (this.isSchedulable(provider)) {
            TimerProvider timer = (TimerProvider)session.getProvider(TimerProvider.class);
            if (timer == null) {
                logger.debugf("Timer provider not available. Not scheduling periodic sync task for provider '%s' in realm '%s'", (Object)provider.getName(), (Object)this.realmId);
                return false;
            }
            logger.debugf("Scheduling user periodic sync task '%s' for user storage provider '%s' in realm '%s' with period %d seconds", new Object[]{this.getTaskName(), provider.getName(), this.realmId, this.period});
            timer.scheduleTask((ScheduledTask)this, (long)this.period * 1000L);
            return true;
        }
        logger.debugf("Not scheduling periodic sync settings for provider '%s' in realm '%s'", (Object)provider.getName(), (Object)this.realmId);
        return false;
    }

    void cancel(KeycloakSession session) {
        TimerProvider timer = (TimerProvider)session.getProvider(TimerProvider.class);
        if (timer == null) {
            logger.debugf("Timer provider not available. Not cancelling periodic sync task for provider id '%s' in realm '%s'", (Object)this.providerId, (Object)this.realmId);
            return;
        }
        UserStorageProviderModel provider = this.getStorageModel(session);
        logger.debugf("Cancelling any running user periodic sync task '%s' for user storage provider provider '%s' in realm '%s'", (Object)this.getTaskName(), (Object)provider.getName(), (Object)this.realmId);
        TimerProvider.TimerTaskContext existingTask = timer.cancelTask(this.getTaskName());
        if (existingTask != null) {
            logger.debugf("Cancelled periodic sync task with task-name '%s' for provider with id '%s' and name '%s'", (Object)this.getTaskName(), (Object)provider.getId(), (Object)provider.getName());
        }
    }

    private UserStorageProviderModel getStorageModel(KeycloakSession session) {
        RealmModel realm = session.getContext().getRealm();
        if (realm == null) {
            throw new ModelIllegalStateException("Realm with id " + this.realmId + " not found");
        }
        ComponentModel component = realm.getComponent(this.providerId);
        if (component == null) {
            this.cancel(session);
            throw new ModelIllegalStateException("User storage provider with id " + this.providerId + " not found in realm " + realm.getName());
        }
        return new UserStorageProviderModel(component);
    }

    private SynchronizationResult runFullSync(KeycloakSession session) {
        return this.runSync(session, (TriFunction<KeycloakSessionFactory, ImportSynchronization, UserStorageProviderModel, SynchronizationResult>)((TriFunction)(sf, storage, model) -> storage.sync(sf, this.realmId, model)));
    }

    private SynchronizationResult runIncrementalSync(KeycloakSession session) {
        return this.runSync(session, (TriFunction<KeycloakSessionFactory, ImportSynchronization, UserStorageProviderModel, SynchronizationResult>)((TriFunction)(sf, storage, model) -> {
            int oldLastSync = model.getLastSync();
            return storage.syncSince(Time.toDate((int)oldLastSync), sf, this.realmId, model);
        }));
    }

    private SynchronizationResult runSync(KeycloakSession session, TriFunction<KeycloakSessionFactory, ImportSynchronization, UserStorageProviderModel, SynchronizationResult> syncFunction) {
        int timeout;
        String taskKey;
        UserStorageProviderModel provider = this.getStorageModel(session);
        KeycloakSessionFactory sessionFactory = session.getKeycloakSessionFactory();
        ImportSynchronization factory = this.getProviderFactory(session, provider);
        if (factory == null) {
            return SynchronizationResult.ignored();
        }
        ClusterProvider clusterProvider = (ClusterProvider)session.getProvider(ClusterProvider.class);
        ExecutionResult task = clusterProvider.executeIfNotExecuted(taskKey = provider.getId() + "::sync", timeout = Math.max(30, this.period), () -> {
            SynchronizationResult result = (SynchronizationResult)syncFunction.apply((Object)sessionFactory, (Object)factory, (Object)provider);
            if (!result.isIgnored()) {
                this.updateLastSyncInterval(session);
            }
            return result;
        });
        SynchronizationResult result = (SynchronizationResult)task.getResult();
        if (result == null || !task.isExecuted()) {
            logger.debugf("syncing users for federation provider %s was ignored as it's already in progress", (Object)provider.getName());
            return SynchronizationResult.ignored();
        }
        return result;
    }

    private ImportSynchronization getProviderFactory(KeycloakSession session, UserStorageProviderModel provider) {
        KeycloakSessionFactory sessionFactory = session.getKeycloakSessionFactory();
        UserStorageProviderFactory factory = (UserStorageProviderFactory)sessionFactory.getProviderFactory(UserStorageProvider.class, provider.getProviderId());
        if (factory instanceof ImportSynchronization) {
            ImportSynchronization f = (ImportSynchronization)factory;
            return f;
        }
        return null;
    }

    private void updateLastSyncInterval(KeycloakSession session) {
        UserStorageProviderModel provider = this.getStorageModel(session);
        provider.setLastSync(Time.currentTime(), this.syncMode);
        RealmModel realm = session.getContext().getRealm();
        realm.updateComponent((ComponentModel)provider);
    }

    private boolean isSyncPeriod(UserStorageProviderModel provider) {
        int lastSyncTime = provider.getLastSync(this.syncMode);
        if (lastSyncTime <= 0) {
            return true;
        }
        int currentTime = Time.currentTime();
        int timeSinceLastSync = currentTime - lastSyncTime;
        return timeSinceLastSync > this.period;
    }

    private boolean isSchedulable(UserStorageProviderModel provider) {
        return provider.isImportEnabled() && provider.isEnabled() && this.period > 0;
    }
}

