﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/ds/DirectoryServiceRequest.h>
#include <aws/ds/DirectoryService_EXPORTS.h>
#include <aws/ds/model/DirectoryVpcSettings.h>

#include <utility>

namespace Aws {
namespace DirectoryService {
namespace Model {

/**
 */
class AddRegionRequest : public DirectoryServiceRequest {
 public:
  AWS_DIRECTORYSERVICE_API AddRegionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AddRegion"; }

  AWS_DIRECTORYSERVICE_API Aws::String SerializePayload() const override;

  AWS_DIRECTORYSERVICE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The identifier of the directory to which you want to add Region
   * replication.</p>
   */
  inline const Aws::String& GetDirectoryId() const { return m_directoryId; }
  inline bool DirectoryIdHasBeenSet() const { return m_directoryIdHasBeenSet; }
  template <typename DirectoryIdT = Aws::String>
  void SetDirectoryId(DirectoryIdT&& value) {
    m_directoryIdHasBeenSet = true;
    m_directoryId = std::forward<DirectoryIdT>(value);
  }
  template <typename DirectoryIdT = Aws::String>
  AddRegionRequest& WithDirectoryId(DirectoryIdT&& value) {
    SetDirectoryId(std::forward<DirectoryIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the Region where you want to add domain controllers for
   * replication. For example, <code>us-east-1</code>.</p>
   */
  inline const Aws::String& GetRegionName() const { return m_regionName; }
  inline bool RegionNameHasBeenSet() const { return m_regionNameHasBeenSet; }
  template <typename RegionNameT = Aws::String>
  void SetRegionName(RegionNameT&& value) {
    m_regionNameHasBeenSet = true;
    m_regionName = std::forward<RegionNameT>(value);
  }
  template <typename RegionNameT = Aws::String>
  AddRegionRequest& WithRegionName(RegionNameT&& value) {
    SetRegionName(std::forward<RegionNameT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const DirectoryVpcSettings& GetVPCSettings() const { return m_vPCSettings; }
  inline bool VPCSettingsHasBeenSet() const { return m_vPCSettingsHasBeenSet; }
  template <typename VPCSettingsT = DirectoryVpcSettings>
  void SetVPCSettings(VPCSettingsT&& value) {
    m_vPCSettingsHasBeenSet = true;
    m_vPCSettings = std::forward<VPCSettingsT>(value);
  }
  template <typename VPCSettingsT = DirectoryVpcSettings>
  AddRegionRequest& WithVPCSettings(VPCSettingsT&& value) {
    SetVPCSettings(std::forward<VPCSettingsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_directoryId;

  Aws::String m_regionName;

  DirectoryVpcSettings m_vPCSettings;
  bool m_directoryIdHasBeenSet = false;
  bool m_regionNameHasBeenSet = false;
  bool m_vPCSettingsHasBeenSet = false;
};

}  // namespace Model
}  // namespace DirectoryService
}  // namespace Aws
